classdef DynamicsInfUpstreamRA < SteadyStateUpstreamRA
    properties (GetAccess = public, SetAccess = private)
        Ct                  % Aggregate Consumption
        Rt                  % Nominal interest rate sequence
        Nt                  % Employment sequence
        At                  % Savings sequence
        wt                  % Nominal Wages
        pt                  % Price level
        qt                  % Asset price
        Deltat              % Price dispersion distorsion
        p_upstrt              % Price index for the intermediate good
        t_change            % Time of the experiment
        pt_realized         % Price level realized
        wt_realized         % Nominal Wage realized
        p_upstrt_realized      % Intermediate prices realized
        taylor_inf = 1.5    % Taylor inflation coefficient
    end
    
    properties (Dependent=true, SetAccess=protected)% play with those if things don't work...
        Yt                  % Aggregate Output time t
        inflt               % Inflation time t
    end
    
    methods
        
        function res = get.Yt(obj)
            res = obj.Nt.^(1-obj.delta)./obj.Deltat;
        end
        
        function res = get.inflt(obj)
            res = [obj.pt_realized(1,:); obj.pt_realized(2:end,:)./obj.pt_realized(1:end-1,:)];
        end
        
        function obj = DynamicsInfUpstreamRA(calibr)
            % Constructor. pass it a calibration structure containing
            % relevant parameters
            obj  = obj@SteadyStateUpstreamRA(calibr);            
            obj.taylor_inf = calibr.taylor_inf;
            disp('Steady state found successfully');
            
        end
        
        function one_period_shock(obj, t_change, shock, T, K)
            K = K+1;
            % Relevant sequences to calculate equilibrium
            obj.Ct = ones(T,K)*obj.C;
            
            obj.Rt = obj.R*ones(T,1);
            obj.Rt(t_change) = obj.Rt(t_change)*shock;
            obj.t_change = t_change;
            
            obj.Nt = ones(T,K)*obj.N;
            
            obj.At = ones(T,K)*obj.A;
            
            obj.wt = ones(T,K)*obj.w;
            obj.wt_realized = ones(T,K)*obj.w;
            
            obj.pt = ones(T,K);
            obj.pt_realized = ones(T,K);
            
            obj.p_upstrt = ones(T,K)*obj.p_upstr;
            obj.p_upstrt_realized = ones(T,K)*obj.p_upstr;
            
            obj.qt = ones(T,K)*obj.q;
            
            obj.Deltat = ones(T,K)*obj.Delta;
            
            for k = 2:K
%                 disp(['K=', num2str(k-1)]);
                
                % Impose Taylor Rule after the experiment
                obj.Rt(t_change+1:end) = obj.R*(obj.pt(t_change+1:end,k-1)./obj.pt(t_change:end-1,k-1)).^obj.taylor_inf;
                
                for t=1:T-1
                    % Calculate inflation (it is predetermined in t-1)
                    cumR = cumprod([1; obj.Rt(t:end-1)]);
                    cumlambda = obj.lambda.^(0:length(cumR)-1);
                    if t==1
                        numerator = obj.p_upstr;
                        denominator = 1;
                    else
                        numerator = cumlambda(1:end-1)./cumR(1:end-1)'*(obj.Yt(t:end-1,k-1).*(obj.pt(t:end-1,k-1)).^obj.epsilon.*obj.p_upstrt(t:end-1,k-1)) + obj.p_upstrt(end,k-1)*obj.pt(end,k-1)^obj.epsilon*obj.Y*obj.R/(obj.R-obj.lambda)*cumlambda(end)/cumR(end);
                        denominator = cumlambda(1:end-1)./cumR(1:end-1)'*(obj.Yt(t:end-1,k-1).*(obj.pt(t:end-1,k-1)).^obj.epsilon) + obj.pt(end,k-1)^obj.epsilon*obj.Y*obj.R/(obj.R-obj.lambda)*cumlambda(end)/cumR(end);
                    end
                    pstar = obj.epsilon/(obj.epsilon-1)*numerator/denominator;
                    if t==1
                        p_last = 1;
                        Delta_last = obj.Delta;
                    else
                        p_last = obj.pt(t-1,k);
                        Delta_last = obj.Deltat(t-1,k);
                    end
					
                    obj.pt(t,k) = (obj.lambda*p_last^(1-obj.epsilon) + (1-obj.lambda)*pstar^(1-obj.epsilon))^(1/(1-obj.epsilon));
                    infl = obj.pt(t,k)/p_last; % Intermediate price index inflation
					
                    if obj.lambda ~= 1
                        obj.Deltat(t,k) = obj.lambda*Delta_last*infl^obj.epsilon + (1-obj.lambda)*(max(1-obj.lambda*infl^(obj.epsilon-1),0)/(1-obj.lambda))^(obj.epsilon/(obj.epsilon-1));
                    else
                        obj.Delta = 1; % Rigid prices
                    end
                    
                    % Impose Taylor Rule after the experiment
                    if t>obj.t_change
                        obj.Rt(t) = obj.R*infl^obj.taylor_inf;
                    end
                    
                    % Update price sequence to let expected inflation to be constant
                    obj.pt(t+1:end,k-1) = obj.pt(t+1:end,k-1)/obj.pt(t,k-1)*obj.pt(t,k);
                    obj.p_upstrt(t+1:end,k-1) = obj.p_upstrt(t+1:end,k-1)/obj.pt(t,k-1)*obj.pt(t,k);
                    
					R_real_loop = obj.Rt(t)/obj.pt(t+1,k-1)*obj.pt(t,k);
					cumR = cumprod(obj.Rt(t:end));
					obj.qt(t,k) = obj.delta*(obj.Yt(t+1:end,k-1)'*(obj.pt(t+1:end,k-1)./cumR(1:end-1)) + obj.Y*obj.pt(end,k-1)*obj.R/(cumR(end)*(obj.R-1)))/obj.pt(t,k);
					cumBeta = cumprod(obj.beta * ones(length(obj.Rt(t:end)), 1));
					
					pv_y = (1-obj.delta)*(obj.Yt(t+1:end,k-1)'*(obj.pt(t+1:end,k-1)./cumR(1:end-1)) + obj.Y*obj.pt(end,k-1)*obj.R/(cumR(end)*(obj.R-1)))/obj.pt(t,k);
					mpc = ((cumBeta(1:end-1)'.^(1/obj.sigma))*((obj.pt(t+1:end,k-1)./obj.pt(t,k)./cumR(1:end-1)).^(1-1/obj.sigma)) + cumBeta(end)^(1/obj.sigma)*((obj.pt(end,k-1)/obj.pt(t,k)/cumR(end))^(1-1/obj.sigma))/(1-obj.beta))^-1;
					obj.Ct(t,k) = mpc*(pv_y + obj.qt(t,k));
					
					obj.Nt(t,k) = obj.Ct(t,k)^(1/(1-obj.delta));
					
					w_real = obj.Nt(t,k)^obj.gamma*obj.Ct(t,k)^obj.sigma/(obj.ssmean^(1+obj.gamma));
					obj.At(t+1,k) = obj.qt(t,k)*R_real_loop;
					
                    obj.wt(t,k) = w_real*obj.pt(t,k);
                    
                    % Upstream good price ends up being
                    obj.p_upstrt(t,k) = obj.wt(t,k)/(1-obj.delta)*obj.Nt(t,k)^obj.delta;
                end
                obj.pt(T,k) = obj.pt(T-1,k);
                obj.wt(T,k) = obj.wt(T-1,k);
                obj.p_upstrt(T,k) = obj.p_upstrt(T-1,k);
                obj.pt_realized(:,k) = obj.pt(:,k);
                obj.wt_realized(:,k) = obj.wt(:,k);
                obj.p_upstrt_realized(:,k) = obj.p_upstrt(:,k);
            end
        end
	end
end