*** 
*** Code for "The Labor Market Integration of Refugee Migrants in High-Income Countries"
*** Courtney Brell, Christian Dustmann, and Ian Preston
***
*** Analysis of compiled labor market moments
*** This file should be run in the folder containing the file refugees_combined.dta
***

********************************************************************************
*** Preliminaries

clear all
use "refugees_combined", clear

********************************************************************************
*** Generate tables

*** Table 1 (employment gaps)
preserve
	* Look at outcomes over all individuals
	keep if missing(female)
	drop if Nemp<40

	* Generate comparison native figures
	replace yearssincearrive=2 if migranttype==0
	expand 2 if migranttype==0, gen(flag)
	replace yearssincearrive=10 if flag==1

	keep if (yearssincearrive==2|yearssincearrive==10)

	* Calculate employment gaps
	keep employment country yearssincearrive migranttype
	reshape wide employment, i(country yearssincearrive) j(migranttype)
	keep if !missing(employment0)&!missing(employment1)&!missing(employment2)
	gen refimmgap=employment2-employment1
	gen refnatgap=employment0-employment1

	* Output data
	sort yearssincearrive country
	list country yearssincearrive employment1 refimmgap refnatgap
restore

**** Table 2 (employment growth)
* Employment growth in first 5 years
preserve
	keep if missing(female)
	drop if migranttype==0
	drop if missing(employment)
	drop if Nemp<40

	* Get first and last entries for 0-5 ysm; calculate average growth
	keep if yearssincearrive<=5
	bysort country migranttype (yearssincearrive): keep if _n==1|_n==_N
	bysort country migranttype (yearssincearrive): gen firstemp=(_n==1)
	bysort country migranttype (yearssincearrive): gen lastemp=(_n==_N)
	bysort country migranttype: egen yeargap=total(lastemp*yearssincearrive-firstemp*yearssincearrive)
	bysort country migranttype: egen empgap=total(lastemp*employment-firstemp*employment)
	duplicates drop country migranttype, force
	gen empgrowthrate=empgap/yeargap

	* Output data
	list country empgrowthrate if migranttype==1
	list country empgrowthrate if migranttype==2
restore
* Employment growth in 5-10 years
preserve
	keep if missing(female)
	drop if migranttype==0
	drop if missing(employment)
	drop if Nemp<40

	* Get first and last entries for 5-10 ysm; calculate average growth
	keep if yearssincearrive>=5&yearssincearrive<=10
	bysort country migranttype (yearssincearrive): keep if _n==1|_n==_N
	bysort country migranttype (yearssincearrive): gen firstemp=(_n==1)
	bysort country migranttype (yearssincearrive): gen lastemp=(_n==_N)
	bysort country migranttype: egen yeargap=total(lastemp*yearssincearrive-firstemp*yearssincearrive)
	bysort country migranttype: egen empgap=total(lastemp*employment-firstemp*employment)
	duplicates drop country migranttype, force
	gen empgrowthrate=empgap/yeargap

	* Output data
	list country empgrowthrate if migranttype==1
	list country empgrowthrate if migranttype==2
restore

*** Table 3 (wage gaps)
preserve
	keep if missing(female)
	drop if Ninc<40

	* Calculate native wages
	bysort country: egen natwage=median(cond(migranttype==0,avg_income,.))
	drop if migranttype==0

	keep if (yearssincearrive==2|yearssincearrive==10)

	* Calculate wage ratios
	gen wageratio=avg_income/natwage
	keep wageratio country yearssincearrive migranttype
	reshape wide wageratio, i(country yearssincearrive) j(migranttype)
	keep if !missing(wageratio1)&!missing(wageratio2)
	gen refimmratio=wageratio1/wageratio2

	* Output data
	sort yearssincearrive country
	list country yearssincearrive wageratio1 refimmratio
restore

*** Table A1 (employment gender ratios)
preserve
	keep if !missing(female)
	drop if Nemp<40

	* Construct native comparisons
	replace yearssincearrive=2 if migranttype==0
	expand 2 if migranttype==0, gen(flag)
	replace yearssincearrive=10 if flag==1

	keep if (yearssincearrive==2|yearssincearrive==10)

	keep employment female country yearssincearrive migranttype
	reshape wide employment, i(country female yearssincearrive) j(migranttype)
	keep if !missing(employment0)&!missing(employment1)&!missing(employment2)
	reshape wide employment0 employment1 employment2, i(country yearssincearrive) j(female)

	* Calculate the gender employment rate ratios
	gen refratio=employment11/employment10
	gen immratio=employment21/employment20
	gen natratio=employment01/employment00

	* Output data
	sort yearssincearrive country
	list country yearssincearrive refratio immratio natratio
restore

********************************************************************************
*** Generate figures

*** Figure 2 (employment)
preserve
	* Select observations of interest
	keep if missing(female)&(migranttype==1|migranttype==2)&yearssincearrive>=0&yearssincearrive<=10
	drop if Nemp<40
	* Generate graph
	local countrylist "Australia Canada Denmark Finland Germany Norway Sweden UK USA"
	local graphlist ""
	foreach loopcountry of local countrylist {
		twoway ///
			(connected employment yearssincearrive if migranttype==1&country=="`loopcountry'", sort) ///
			(connected employment yearssincearrive if migranttype==2&country=="`loopcountry'", sort) ///
			, legend(label(1 "Refugees") label(2 "Other immigrants")) xtitle("Years since migration") ytitle("Employment rate") name(g`loopcountry', replace) subtitle("`loopcountry'") nodraw
		local graphlist "`graphlist' g`loopcountry'"
	}
	graph combine `graphlist', col(3) ycommon xcommon
	graph save "employment", replace
restore

** Figure 4 (wages)
preserve
	* Generate native average wage
	bysort country: egen natwage=median(cond(migranttype==0&missing(female),avg_income,.))
	gen wageratio=avg_income/natwage
	* Select observations of interest
	keep if missing(female)&(migranttype==1|migranttype==2)&yearssincearrive>=0&yearssincearrive<=10
	drop if Ninc<40
	* Generate graph
	local countrylist "Australia Canada Finland Germany Norway Sweden USA"
	local graphlist ""
	foreach loopcountry of local countrylist {
		twoway ///
			(connected wageratio yearssincearrive if migranttype==1&country=="`loopcountry'", sort) ///
			(connected wageratio yearssincearrive if migranttype==2&country=="`loopcountry'", sort) ///
			, legend(label(1 "Refugees") label(2 "Other immigrants")) xtitle("Years since migration") ytitle("Average wage ratio to natives") name(g`loopcountry', replace) subtitle("`loopcountry'") nodraw
		local graphlist "`graphlist' g`loopcountry'"
	}
	graph combine `graphlist', col(3) ycommon xcommon
	graph save "wages", replace
restore

*** Figure A1 (employment by gender)
preserve
	* Select observations of interest
	keep if (female==0|female==1)&migranttype==1&yearssincearrive>=0&yearssincearrive<=10
	drop if Nemp<40
	* Generate graph
	local countrylist "Australia Canada Denmark Finland Germany Norway Sweden UK USA"
	local graphlist ""
	foreach loopcountry of local countrylist {
		twoway ///
			(connected employment yearssincearrive if female==0&country=="`loopcountry'", sort) ///
			(connected employment yearssincearrive if female==1&country=="`loopcountry'", sort) ///
			, legend(label(1 "Male refugees") label(2 "Female refugees")) xtitle("Years since migration") ytitle("Employment rate") name(g`loopcountry', replace) subtitle("`loopcountry'") nodraw
		local graphlist "`graphlist' g`loopcountry'"
	}
	graph combine `graphlist', col(3) ycommon xcommon
	graph save "employment_bygender", replace
restore

********************************************************************************
*** Clean up

clear all
